/////////////////////////////////////////////////////////////////////////////////
//
// License Creative Commons Attribution-NonCommercial-ShareAlike 3.0 Unported License.
//
/////////////////////////////////////////////////////////////////////////////////
//
// This shader is based on Huw Bowles'  FSS,  Fast Adaptive Stationary Sampling,
// shader downloaded from ShaderToy.com and is therfore subject  to  ShaderToy's
// licence conditions, see above. I have slightly modified the shader by
//
//   1) Adapting it to run with the VGHD software.
//
//   2) Replacing the texture based noise function with a purely
//      procedual noise function taken from Alexander Alekseev's
//      Seascape shader, also obtained from ShaderToy.com.
//
//   3) Negating the Y co-ordinate of the vector q  declared  at
//      the start of the main routine. This was done so that the
//      clip's image used as the main texture was upright.
//
//   4) Modifying the main procedure so that the generated smoke
//      or fog overlays the image used as the texture input.
//
//   5) Tidied up the code in a couple of places.
//
// Modification 2 was a lucky accident. I had mistakenly thought that the origin
// was the center of the screen so negating Y would properly flip the image  and
// have no effect on the shader.  However,  the origin is the top left corner of
// the screen and the proper tranformation was y -> 1 - y.  Never-the-less,  the 
// incorrect tranformation worked but it also had the side  effect  of  changing 
// the appearance of the shader's output from a cloud bank to a smoke cloud.
//
// Modification 6 allows the shader to be used with a background scene.
//
/////////////////////////////////////////////////////////////////////////////////

// The following set of parameters control the properties of the cloud  and  can
// be freely modified to suit your particular taste.

uniform vec4 cloudiness;   // Currently only first 3 components used.
uniform vec2 cloud_level;  // Both elements are used.

// CLOUD_DENSITY controls how thick the cloud appears to be.  It can take values
// ranging from 0.0, for no cloud, up to 1.0 for the thickest cloud.

#define CLOUD_DENSITY cloudiness.x

// CLOUD_QUALITY controls the general apearance of the cloud.  It  has no simple
// interpretation but,  if  the shader's internal parameters are left unchanged, 
// the following observations can be made
//
//    0.0 - no internal structure
//    2.0 - internal billows begin to appear
//    3.0 - stronger internal billowing
//    4.0 - internal cloud plumes begin to appear
//    7.0 - stronger internal cloud plumes
//    9.0 - internal plumes begin to break up
//   11.0 - cloud plumes begin to be clumpy
//   13.0 - increased cloud clumpiness
//
// as the values grow even higher the "clumpiness" increases and the size of the
// clumps become smaller. At 100.0 the resmblence to cloud has been lost and the
// effect is more like a snow, but with the snow rising not falling.

#define CLOUD_QUALITY cloudiness.y

// CLOUD_FADE_RATE controls the rate at which the cloud fades as it goes up  the
// screen. A value of 1.0 give linear fading, 2.0 quadratic etc., so high values
// cause the cloud to fade very rapidly leaving clear air above or below a  pool
// of cloud with a distinct boundary while lower layers cause there to be a less
// distinct boundary.

#define CLOUD_FADE_RATE cloudiness.z // Controls rate of fading in Y direction

// CLOUD_LEVEL_0 and CLOUD_LEVEL_1 define the heights at which the cloud density
// are at its minimum and maximum values.  Their ranges are both from 0.0 to 1.0
// with 0.0 being the bottom of the screen and 1.0 the top. Between these levels
// the density of the increases from zero at CLOUD_LEVEL_0 to its maximum  value
// at CLOUD_LEVEL_1.

#define CLOUD_LEVEL_0 cloud_level.x // Height at which cloud density minimum.
#define CLOUD_LEVEL_1 cloud_level.y // Height at which cloud density maximum.

// CLOUD_WAVE_F, CLOUD_WAVE_X and CLOUD_WAVE_H control the waves on the  surface
// of the cloud pool. These waves do propagate into the cloud above the pool but
// are attenuated with height along with the rest of the cloud.
//
// It ought to be possible to pass these values in from the  .scn  file  in  the
// same way as the "cloudiness" and "cloud_level" are passed, but whenever I try
// to do it it causes a dark rectangle to appear about the performer.

#define CLOUD_WAVE_HEIGHT 0.1 // cloud_wave.x // Wave height
#define CLOUD_WAVE_F_T    2.0 // cloud_wave.y // Wave frequency in time
#define CLOUD_WAVE_F_X    9.0 // cloud_wave.z // Wave frequency in X direction

/////////////////////////////////////////////////////////////////////////////////

// Shader inputs.

uniform float u_Elapsed;    // The elapsed time in seconds
uniform vec2  u_WindowSize; // Window dimensions in pixels

// Use defines here rather than edit the body of the code.

#define iGlobalTime u_Elapsed
#define iResolution u_WindowSize

varying vec4 gl_TexCoord[];
varying vec4 gl_Color;

uniform sampler2D SourceImage;

/////////////////////////////////////////////////////////////////////////////////

// Internal parameters. Only change these if you want to play with the internal
// operation of the shader itself rather than just tweaking the cloud. TheEmu.

#define SAMPLE_COUNT 32
#define DIST_MAX 128.
#define SAMPLES_ADAPTIVITY .2
#define EPS 0.001

// cam moving in a straight line

const vec3 lookDir = vec3(-1.,0.,0.);
const vec3 camVel  = vec3(-0.1,0.,0.);

float samplesCurvature = 0.8; // mix between fixed z and fixed radius sampling

vec3 sundir = normalize(vec3(-1.0,0.0,1.0));

// ----------------------------------------------------------------------------

// Original texture based noise function replaced by a purely  procedural
// noise function taken from Alexander Alekseev's Seascape  shader.  This
// is neccessary with the current version of VGHD as  it  seems  able  to
// only provide a single texture as input and using the clipSprite as the
// noise texture led to visible artifacts in the final scene. TheEmu.

float hash( vec2 p ) {
    float h = dot(p,vec2(127.1,311.7));
    return fract(sin(h)*43758.5453123);
}

float noise( in vec3 p ) {
    vec2 i = floor( p.xz );
    vec2 f = fract( p.xz );
    vec2 u = f*f*(3.0-2.0*f);
    return -1.0+2.0*mix( mix( hash( i + vec2(0.0,0.0) ),
                     hash( i + vec2(1.0,0.0) ), u.x),
                mix( hash( i + vec2(0.0,1.0) ),
                     hash( i + vec2(1.0,1.0) ), u.x), u.y);
}

// ----------------------------------------------------------------------------

vec4 map( in vec3 p )
{
    float d = 0.2 - p.y;

    vec3 q = p*CLOUD_QUALITY;

    float f  = 0.5000*noise( q ); q = q*2.02;
             + 0.2500*noise( q ); q = q*2.03;
             + 0.1250*noise( q ); q = q*2.01;
             + 0.0625*noise( q );

    d += 3.0 * f;

    d = clamp( d, 0.0, 1.0 );

    vec4 res = vec4( d );

    res.xyz = mix( 1.15*vec3(1.0,0.95,0.8), vec3(0.7,0.7,0.7), res.x );

    return res;
}

// ----------------------------------------------------------------------------

float spacing(float t )
{
    t = max(t,0.0);

    // unnorm pdf - plot this in graphtoy to see shape
    float pdf = 1. / (SAMPLES_ADAPTIVITY*t + 1.);

   // integral of pdf over dist
   float norm = (1. / SAMPLES_ADAPTIVITY)*log(1. + SAMPLES_ADAPTIVITY*DIST_MAX);
    // norm pdf
    pdf /= norm;

    // sample spacing for our sample count
    return 1. / (float(SAMPLE_COUNT) * pdf);
}

// ----------------------------------------------------------------------------

void firstT( out float t, out float dt, out float wt, out bool even )
{
    dt = exp2(floor(log2(spacing(0.))));
    t = dt - mod(t,dt);
    even = mod(t,2.*dt) < EPS;
    wt = 1.;
}

// ----------------------------------------------------------------------------

void nextT( inout float t, inout float dt, inout float wt, inout bool even )
{
    float s = spacing(t); // get desired sample spacing

    if( s < dt )
      { // Immediately move to higher density
        dt /= 2.;
        even = true;
      } 
    else if( even && s > 2.*dt )
      { // Move to lower density if a sample is there
        dt *= 2.;
        wt = 1.;
        even = mod(t,2.*dt) < EPS;
      } 

    if( even )
     { // Update sample based on how far we are into its band
      wt = clamp( 2. - s/dt,0.,1.);
     } 

    // next sample

    t += dt;
    even = !even;

}

// ----------------------------------------------------------------------------

float sampleWt( float wt, bool even )
{
    return even ? (2.-wt) : wt;
}

// ----------------------------------------------------------------------------

vec4 raymarch( in vec3 ro, in vec3 rd )
{
    vec4 sum = vec4(0, 0, 0, 0);

    // Setup sampling

    float t, dt, wt; bool even;
    firstT( t, dt, wt, even );

    for(int i=0; i<SAMPLE_COUNT; i++)
    {
        if( sum.a > 0.99 ) continue;

        vec3 pos = ro + t*rd;
        vec4 col = map( pos );

        // Iq's goodness

        float dif =  clamp((col.w - map(pos+0.3*sundir).w)/0.6, 0.0, 1.0 );
        vec3 lin = vec3(0.65,0.68,0.7)*1.35 + 0.45*vec3(0.7, 0.5, 0.3)*dif;

        col.xyz *= lin;
        col.a   *= 0.35;
        col.rgb *= col.a;

        // Fade samples at far field
        // .3 is an ugly fudge factor due to oversampling

        float fadeout = 1.-clamp((t/(DIST_MAX*.3)-.85)/.15,0.,1.); 

        // Integrate

        float thisDt = dt * sampleWt( wt, even); // blend in dts
        thisDt = sqrt(thisDt/5.0 )*5.0;          // hack to soften and brighten

        sum += thisDt * col * (1.0 - sum.a) * fadeout;

        // Next sample

        nextT( t, dt, wt, even );
    }

    sum.xyz /= (0.001+sum.w);

    return clamp( sum, 0.0, 1.0 );
}

// ----------------------------------------------------------------------------

vec4 cloud ( vec2 q )
{
    // "Camera" for the cloud generation.

    vec2 p = 2.0*q - 1.0;

    vec3 ro = vec3(0.,1.9,0.) + iGlobalTime*camVel;

    vec3 ww = normalize(lookDir);
    vec3 uu = normalize(cross( vec3(0.0,1.0,0.0), ww ));
    vec3 vv = normalize(cross(ww,uu));
    vec3 rd = normalize( p.x*uu + 1.2*p.y*vv + 1.5*ww );

    // Generate the cloud using Huw Bowles's FSS method.

    vec4 res = raymarch( ro, rd/mix(dot(rd,ww),1.,samplesCurvature) );

    float sun = clamp( dot(sundir,rd), 0.0, 1.0 );
    vec3 cloud_color = vec3(0.6,0.71,0.75) - rd.y*0.2*vec3(1.0,0.5,1.0) + 0.15*0.5;

    cloud_color += 0.2*vec3(1.0,.6,0.1)*pow(sun,8.0);
    cloud_color *= 0.95;
    cloud_color = mix( cloud_color, res.xyz, res.w );
    cloud_color += 0.1*vec3(1.0,0.4,0.2)*pow(sun,3.0);

    // Generate a wave that will be used to adjust the height at  which
    // the cloud has it maximum density and above which  it  begins  to
    // thin out.  Both simple traveling waves sin(fx+fx) and stationary
    // waves, sin(fx*ft), look too regular but the form used here gives
    // nice almost natural looking billows to the cloud surface.

    float ft = iGlobalTime * CLOUD_WAVE_F_T;
    float fx = q.x * CLOUD_WAVE_F_X ;
    float wave = CLOUD_WAVE_HEIGHT * sin(ft+sin(fx));

    // Determine the cloud's weighting factor to be used when combining
    // the cloud with the main texture,  which,  for VGHD,  may  be the
    // instantaneous image from a performer's clip,  a  plain quad or a
    // a static image texture. The weighting factor will be zero at a Y
    // value of CLOUD_LEVEL_0, remember that q.y is negative, rising to
    // its maximum value at CLOUD_LEVEL_1.

    float f;
    
    if ( CLOUD_LEVEL_0 > CLOUD_LEVEL_1 )
     {
       f = ( CLOUD_LEVEL_0 + q.y );
       f = f / ( CLOUD_LEVEL_0 - CLOUD_LEVEL_1 + wave*0.5 );
     }
    else
     {
       f = ( CLOUD_LEVEL_1 + q.y );
       f = f / ( CLOUD_LEVEL_1 - CLOUD_LEVEL_0 + wave*0.5 );
     }

    f = clamp(f,0.0,1.0);
    f = clamp ( pow(f,CLOUD_FADE_RATE), 0.0, 1.0 );
    f = clamp ( f * CLOUD_DENSITY, 0.0, 1.0 );

    return vec4 ( cloud_color, f );

}

// ----------------------------------------------------------------------------

void main(void)
{
    // Get the source image pixel's colour. 

    vec4 image = texture2D(SourceImage,gl_TexCoord[0].xy);

    vec2 q = gl_FragCoord.xy / iResolution.xy;
    q.y = - q.y; // Gets the clip rendered the right way up.

    // Apply the cloud to the scene using the weighting factor.

    vec4 h = cloud(q);

    gl_FragColor = vec4( mix(image.rgb,h.rgb,h.a), 1.0 ) * gl_Color;

}

// ----------------------------------------------------------------------------
